/************************************************************************************************************\

Module Name:    RtspServerModule.c

Description:    .

    Copyright (c) 2015, Matrox Graphics Inc. All Rights Reserved.

    BSD 2-Clause License

    Redistribution and use in source and binary forms, with or without modification, are permitted provided
    that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this list of conditions and the
       following disclaimer.

    2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and
       the following disclaimer in the documentation and/or other materials provided with the distribution.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
    WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
    PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
    ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
    LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
    INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
    TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
    ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

\************************************************************************************************************/

// -----------------------------------------------------------------------------------------------------------
//                                  I N C L U D E S   A N D   U S I N G S
// -----------------------------------------------------------------------------------------------------------

#include "RtspServerModule.h"

// -----------------------------------------------------------------------------------------------------------
//                         N A M E S P A C E ,   C O N S T A N T S   A N D   T Y P E S
// -----------------------------------------------------------------------------------------------------------

typedef struct tagStrmThreadData
{
    RtspServerModule*   poRtspSrvMod;
    LNetStreamer_Handle hSession;
    MBOOL32             bIsMainThread;
    MBOOL32             bIsVideoThread;

} StrmThreadData;

enum
{
    NaluFlag_Slice  = (1 << LH264_NaluType_SLICE),
    NaluFlag_Idr    = (1 << LH264_NaluType_IDR),
    NaluFlag_Sps    = (1 << LH264_NaluType_SPS),
    NaluFlag_Pps    = (1 << LH264_NaluType_PPS),
};

// -----------------------------------------------------------------------------------------------------------
//                        S T A T I C   M E M B E R S   I N I T I A L I S A T I O N
// -----------------------------------------------------------------------------------------------------------

static const    MCHAR8      g_szModuleNameBase[]    = "RtspSrv";
static          MUINT32     g_uiRtspSrvModCount     = 0;

// -----------------------------------------------------------------------------------------------------------
//                                                  C O D E
// -----------------------------------------------------------------------------------------------------------

/************************************************************************************************************\

Function:       RtspSrvMod_Init

Description:    .

\************************************************************************************************************/
LStatus RtspSrvMod_Init(
            RtspServerModule*       poRtspSrvMod,
            LDevice_Handle          hDevice)
{
    MsgLog(2, "{...");

    RtspSrvMod_Cleanup(poRtspSrvMod);

    LStatus eStatus = ((poRtspSrvMod != MNULL) && (hDevice != MNULL))
                        ? LStatus_OK : LStatus_INVALID_PARAM;

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        poRtspSrvMod->hDevice = hDevice;
        poRtspSrvMod->bUnresponsive = MFALSE;
        poRtspSrvMod->szAddress[0] = '\0';
    }
    else
    {
        RtspSrvMod_Cleanup(poRtspSrvMod);
    }

    MsgLog(2, "...} (status= %d - %s)", eStatus, GetStatusStr(eStatus));

    return eStatus;
}

/************************************************************************************************************\

Function:       RtspSrvMod_Cleanup

Description:    .

\************************************************************************************************************/
void RtspSrvMod_Cleanup(RtspServerModule* poRtspSrvMod)
{
    MsgLog(2, "{...");

    poRtspSrvMod->hDevice               = MNULL;
    poRtspSrvMod->bInit                 = MFALSE;
    poRtspSrvMod->bSessionStarted       = MFALSE;
    poRtspSrvMod->oVideoInLink.poModLnk = MNULL;
    poRtspSrvMod->oAudioInLink.poModLnk = MNULL;
    poRtspSrvMod->uiAudioSpecificConfig = 0;

    MsgLog(2, "...}");
}

/************************************************************************************************************\

Function:       RtspSrvMod_SetParams

Description:    .

\************************************************************************************************************/
LStatus RtspSrvMod_SetParams(
    RtspServerModule*       poRtspSrvMod,
    MUINT                   uiMaxNaluSizeBytes,
    MUINT                   uiMaxAudioBufSizeBytes,
    const char*             szFolderName,
    const char*             szAddress,
    MBOOL                   bEnableVideo,
    MUINT                   uiVideoPort,
    MBOOL                   bEnableAudio,
    MUINT                   uiAudioPort,
    MUINT16                 uiAudioSpecificConfig,
    LNetStreamer_Protocol   eProtocol,
    const char*             szRtmpPushLocation,
    const char*             szUsername,
    const char*             szPassword,
    MUINT                   uiMtu,
    MBOOL                   bEnableSrt,
    MUINT                   uiHlsDuration,
    MUINT                   uiHlsCount,
    const char*             szHlsFolder,
    const char*             szHlsMaster,
    const char*             szNetInterface,
    MBOOL                   bEnableIpv6,
    LNetStreamer_SrtMode    eSrtMode)
{
    LStatus eStatus = LStatus_OK;

    eStatus = ((poRtspSrvMod != MNULL)
               && (!bEnableVideo || (uiMaxNaluSizeBytes > 0))
               && (!bEnableAudio || (uiMaxAudioBufSizeBytes > 0))
               && (szFolderName != MNULL)
               && (bEnableVideo || bEnableAudio))
              ? LStatus_OK : LStatus_INVALID_PARAM;

    if(LSTATUS_IS_SUCCESS(eStatus))
    {
        if (eProtocol == LNetStreamer_Protocol_TS)
        {
            // Only video port is taken into account with TS.
            if (!bEnableVideo)
            {
                uiVideoPort = uiAudioPort;
            }

            uiAudioPort = 0;
        }

        poRtspSrvMod->bEnableAudio              = bEnableAudio;
        poRtspSrvMod->bEnableVideo              = bEnableVideo;
        poRtspSrvMod->uiVideoPort               = uiVideoPort;
        poRtspSrvMod->uiAudioPort               = uiAudioPort;
        poRtspSrvMod->uiMaxAudioBufSizeBytes    = bEnableAudio ? uiMaxAudioBufSizeBytes : 0;
        poRtspSrvMod->uiMaxNaluSizeBytes        = bEnableVideo ? uiMaxNaluSizeBytes : 0;
        poRtspSrvMod->eProtocol                 = eProtocol;
        poRtspSrvMod->uiAudioSpecificConfig     = uiAudioSpecificConfig;

        strncpy_wz(poRtspSrvMod->szFolderName, szFolderName, sizeof(poRtspSrvMod->szFolderName));

        snprintf(
                    poRtspSrvMod->szModuleName,
                    sizeof(poRtspSrvMod->szModuleName),
                    "%s%d",
                    g_szModuleNameBase,
                    g_uiRtspSrvModCount);

        if (szAddress)
        {
            strncpy_wz(poRtspSrvMod->szAddress, szAddress, sizeof(poRtspSrvMod->szAddress));
        }
        else
        {
            poRtspSrvMod->szAddress[0] = 0;
        }

        if (szRtmpPushLocation)
        {
            strncpy_wz(poRtspSrvMod->szRtmpPushLocation, szRtmpPushLocation, sizeof(poRtspSrvMod->szRtmpPushLocation));
        }
        else
        {
            poRtspSrvMod->szRtmpPushLocation[0] = 0;
        }

        if (szUsername)
        {
            strncpy_wz(poRtspSrvMod->szUsername, szUsername, sizeof(poRtspSrvMod->szUsername));
        }
        else
        {
            poRtspSrvMod->szUsername[0] = 0;
        }

        if (szPassword)
        {
            strncpy_wz(poRtspSrvMod->szPassword, szPassword, sizeof(poRtspSrvMod->szPassword));
        }
        else
        {
            poRtspSrvMod->szPassword[0] = 0;
        }

        poRtspSrvMod->uiMtu = uiMtu;

        poRtspSrvMod->bEnableSrt = bEnableSrt;

        poRtspSrvMod->uiHlsDuration = uiHlsDuration;

        poRtspSrvMod->uiHlsCount = uiHlsCount;

        if (szHlsFolder)
        {
            strncpy_wz(poRtspSrvMod->szHlsFolder, szHlsFolder, sizeof(poRtspSrvMod->szHlsFolder));
        }
        else
        {
            poRtspSrvMod->szHlsFolder[0] = 0;
        }

        if (szHlsMaster)
        {
            strncpy_wz(poRtspSrvMod->szHlsMaster, szHlsMaster, sizeof(poRtspSrvMod->szHlsMaster));
        }
        else
        {
            poRtspSrvMod->szHlsMaster[0] = 0;
        }

        if (szNetInterface)
        {
            strncpy_wz(poRtspSrvMod->szNetInterface, szNetInterface, sizeof(poRtspSrvMod->szNetInterface));
        }
        else
        {
            poRtspSrvMod->szNetInterface[0] = 0;
        }

        poRtspSrvMod->bEnableIpv6 = bEnableIpv6;

        poRtspSrvMod->eSrtMode = eSrtMode;

        poRtspSrvMod->bInit = MTRUE;
    }

    ++g_uiRtspSrvModCount;

    return eStatus;
}

/************************************************************************************************************\

Function:       RtspSrvMod_GetNetBuffer

Description:    .

\************************************************************************************************************/
LStatus RtspSrvMod_GetNetBuffer(
        LNetStreamer_Handle     hSession,
        LNetStreamer_Media*     poBufferInfo,
        MUINT                   uiTimeoutMsec)
{
    LStatus eStatus = LStatus_OK;

    MUINT64 uiTimeoutTime = GetMonoTimeUsec() + (uiTimeoutMsec * 1000);

    // Use uiDataMaxLength field to know if the buffer is already acquired
    // ('== 0': Not acquired, '!= 0': Acquired).
    while (poBufferInfo->uiDataMaxLength == 0)
    {
        eStatus = LNetStreamer_GetMedia(hSession, &poBufferInfo->eMediaType);

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            poBufferInfo->uiDataLength = 0;

            if (poBufferInfo->uiDataMaxLength == 0)
            {
                MsgLogErr("ERROR! Acquired buffer is zero size!");
                eStatus = LStatus_FAIL;
            }

            break;
        }
        else
        {
            poBufferInfo->uiDataMaxLength = 0;

            if (GetMonoTimeUsec() > uiTimeoutTime)
            {
                eStatus = LStatus_TIMEOUT;
                break;
            }

            usleep(1000);
        }
    }

    return eStatus;
}

/************************************************************************************************************\

Function:       RtspSrvMod_FillNetBuffer

Description:    .

\************************************************************************************************************/
LStatus RtspSrvMod_FillNetBuffer(
        BufferInfo*             poBuffer,
        LNetStreamer_Media*     poNetBuffer,
        MFLAG32*                pflNaluFlags)
{
    if (poBuffer->hBuffer == MNULL)
    {
        return LStatus_INVALID_PARAM;
    }

    if (poBuffer->uiSizeBytes > g_uiMaxNaluSize)
    {
        MsgLogErr("WARNING! Buffer size greater than %u (size= %lu)",
                  g_uiMaxNaluSize, poBuffer->uiSizeBytes);
    }

    MUINT8* puiBuf = MNULL;
    LStatus eStatus = LBuffer_BeginAccess(poBuffer->hBuffer, 0, 1, &puiBuf);

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        puiBuf += poBuffer->uiStartOffset;

        if ((poNetBuffer->uiDataMaxLength - poNetBuffer->uiDataLength) >= poBuffer->uiSizeBytes)
        {
            memcpy(poNetBuffer->paucData + poNetBuffer->uiDataLength,
                   puiBuf,
                   poBuffer->uiSizeBytes);

            poNetBuffer->uiDataLength += poBuffer->uiSizeBytes;

            if (poNetBuffer->eMediaType == LNetStreamer_MediaType_VIDEO)
            {
                *pflNaluFlags |= (1 << poBuffer->eNaluType);
            }
        }
        else
        {
            MsgLogErr("ERROR! NALU size (%u+%u) larger than maximum size (%u). Skip this NALU...",
                      poNetBuffer->uiDataLength, poBuffer->uiSizeBytes, poNetBuffer->uiDataMaxLength);
        }

        LBuffer_EndAccess(poBuffer->hBuffer);
    }

    return eStatus;
}

/************************************************************************************************************\

Function:       RtspSrvMod_StreamingThread

Description:    .

\************************************************************************************************************/
LStatus RtspSrvMod_StreamingThread(void* pvData)
{
    StrmThreadData*     poThreadData    = (StrmThreadData*) pvData;
    LNetStreamer_Handle hSession        = poThreadData->hSession;
    RtspServerModule*   poRtspSrvMod    = poThreadData->poRtspSrvMod;
    LStatus             eStatus         = LStatus_OK;
    ModuleThread*       poThread        = poThreadData->bIsMainThread
                                          ? &(poRtspSrvMod->oMainCpuThread)
                                          : &(poRtspSrvMod->oChildCpuThread);
    ModuleLinkInput*    poInLink        = poThreadData->bIsVideoThread
                                          ? &(poRtspSrvMod->oVideoInLink)
                                          : &(poRtspSrvMod->oAudioInLink);

    MFLAG32 flNaluTypes = 0;
    LNetStreamer_MediaType eMediaType   = poThreadData->bIsVideoThread
                                          ? LNetStreamer_MediaType_VIDEO
                                          : LNetStreamer_MediaType_AUDIO;
    LNetStreamer_Media  oBufferInfo;
    oBufferInfo.uiDataMaxLength = 0;
    oBufferInfo.eMediaType      = eMediaType;

    MUINT32 uiPrevTsMsec        = 0;
    MUINT32 uiTimeOffsetMsec    = 8*60;
    MUINT32 uiTsAdjMsec         = 0;

    if(poThreadData->bIsVideoThread)
    {
        static MUINT32  uiRtspSrvIndex = 0;

        MCHAR8  szThreadName[16] = "";

        snprintf(szThreadName, sizeof(szThreadName), "RtspSrvVideo%d", uiRtspSrvIndex++);

        ModThread_SetName(szThreadName);
    }
    else
    {
        static MUINT32  uiRtspASrvIndex = 0;

        MCHAR8  szThreadName[16] = "";

        snprintf(szThreadName, sizeof(szThreadName), "RtspSrvAudio%d", uiRtspASrvIndex++);

        ModThread_SetName(szThreadName);
    }

    if(!poThreadData->bIsMainThread)
    {
        MsgLog(2, "Start thread %p", pthread_self());
    }

    MBOOL bSessionInit = MFALSE;

    while (!poThread->bKillThread)
    {
        BufferInfo* poSrcBuffer = MNULL;

        eStatus  = ModLnkIn_GetSubmittedBuffer(
                        poInLink,
                        100,
                        0,
                        MNULL,
                        &poSrcBuffer,
                        MNULL,
                        MNULL);

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            if (poSrcBuffer->bEndOfStream)
            {
                MsgLog(4, "END-OF-STREAM");

                poThread->bKillThread = MTRUE;
            }
            else
            {
                MUINT32 uiTimestampMsec = poSrcBuffer->uiSyncPtsUsec / 1000;

                // Patch to minimize additionnal latency in LNetStreamer: reduce the gap between each timestamps
                // at the beginning of the stream by approximately half of the buffer duration.
                if (bSessionInit && (uiTimeOffsetMsec >= uiTsAdjMsec))
                {
                    if (uiPrevTsMsec != uiTimestampMsec)
                    {
                        if ((uiTsAdjMsec == 0) && (uiPrevTsMsec != 0))
                        {
                            uiTsAdjMsec = (uiTimestampMsec - uiPrevTsMsec + 1) / 2;
                            if(uiTsAdjMsec > 30)
                            {
                                // Discard large gaps.
                                uiTsAdjMsec = 0;
                            }
                        }

                        uiPrevTsMsec      = uiTimestampMsec;
                        uiTimeOffsetMsec -= uiTsAdjMsec;
                    }

                    uiTimestampMsec += uiTimeOffsetMsec;

                    MsgLog(4, "[%c] Adjusted TS= %u msec (offset= %u, adj= %u)",
                           poThreadData->bIsVideoThread ? 'V' : 'A', uiTimestampMsec, uiTimeOffsetMsec, uiTsAdjMsec);
                }

                MsgLog(6, "RtspSrvMod_SendBuffer(%s Buffer[%u], Offset= %u, Size= %u, Timestamp= %u)...",
                            poThreadData->bIsVideoThread ? "Video" : "Audio",
                            poSrcBuffer->uiId,
                            poSrcBuffer->uiStartOffset,
                            poSrcBuffer->uiSizeBytes,
                            uiTimestampMsec);

                while (!poRtspSrvMod->bUnresponsive && !poThread->bKillThread)
                {
                    MUINT64 uiCurTimeUsec = GetMonoTimeUsec();

                    eStatus = RtspSrvMod_GetNetBuffer(hSession, &oBufferInfo, 100);

                    if (LSTATUS_IS_SUCCESS(eStatus))
                    {
                        if (poSrcBuffer != MNULL)
                        {
                            // Only accept video SPS and PPS until session is initialized.
                            if (bSessionInit
                               || ((eMediaType == LNetStreamer_MediaType_VIDEO)
                                   && ((poSrcBuffer->eNaluType == LH264_NaluType_SPS)
                                    || (poSrcBuffer->eNaluType == LH264_NaluType_PPS))))
                            {
                                eStatus = RtspSrvMod_FillNetBuffer(poSrcBuffer, &oBufferInfo, &flNaluTypes);

                                if (LSTATUS_IS_FAIL(eStatus))
                                {
                                    MsgLogErr("ERROR! RtspSrvMod_FillNetBuffer returned %d - %s.",
                                              eStatus, GetStatusStr(eStatus));
                                    break;
                                }
                            }

                            // Return buffer as soon as possible.
                            ModLnkIn_ReturnBuffer(poInLink, poSrcBuffer, MNULL, NO_TAG);
                            poSrcBuffer = MNULL;
                        }

                        if (LSTATUS_IS_SUCCESS(eStatus))
                        {
                            MBOOL bRelease = MTRUE;

                            if (eMediaType == LNetStreamer_MediaType_VIDEO)
                            {
                                bRelease = bSessionInit
                                            ? ((flNaluTypes & NaluFlag_Slice) || (flNaluTypes & NaluFlag_Idr))
                                            : ((flNaluTypes & NaluFlag_Sps)   && (flNaluTypes & NaluFlag_Pps));
                            }

                            if (bRelease)
                            {
                                if (!bSessionInit && (eMediaType == LNetStreamer_MediaType_VIDEO))
                                {
                                    // Flush and stub input queue until session is initialized.
                                    ModLnkIn_CancelSubmittedBuffers(poInLink);
                                    poInLink->poModLnk->abStubOutput[poInLink->uiSubmitIdx] = MTRUE;
                                }

                                oBufferInfo.uiDTS = uiTimestampMsec;
                                oBufferInfo.uiPTS = uiTimestampMsec;

                                eStatus = LNetStreamer_ReleaseMedia(
                                                hSession,
                                                &oBufferInfo.eMediaType);

                                if (LSTATUS_IS_SUCCESS(eStatus))
                                {
                                    oBufferInfo.uiDataMaxLength = 0;
                                    flNaluTypes = 0;

                                    // Session cannot be init if eStatus == LStatus_OK_MISSING_FIRST_SPS_PPS.
                                    if (eStatus == LStatus_OK)
                                    {
                                        bSessionInit = MTRUE;
                                    }
                                }
                            }
                        }
                    }

                    if (eStatus != LStatus_TIMEOUT)
                    {
                        break;
                    }

                    if ((GetMonoTimeUsec() - uiCurTimeUsec) > 5*1000*1000)
                    {
                        poRtspSrvMod->bUnresponsive = MTRUE;
                    }
                    else
                    {
                        MsgLog(0, "RtspSrvMod_SendBuffer TIMEOUT! Retry...");
                    }
                }

                MsgLog(6, "RtspSrvMod_SendBuffer() done. (status= %d - %s)",
                          eStatus, GetStatusStr(eStatus));

                if (poThreadData->bIsMainThread
                     && bSessionInit
                     && !poRtspSrvMod->bSessionStarted
                     && LSTATUS_IS_SUCCESS(eStatus))
                {
                    MUINT32 uiCfgLen     = 0;
                    LStatus eInnerStatus = LNetStreamer_GetConfigLength(
                                                hSession,
                                                LNetStreamer_ConfigType_STANDARD,
                                                &uiCfgLen);

                    if (LSTATUS_IS_SUCCESS(eInnerStatus))
                    {
                        LNetStreamer_Config* poSessionCfg
                            = (LNetStreamer_Config*)malloc(uiCfgLen);

                        if (poSessionCfg != MNULL)
                        {
                            poSessionCfg->eConfigType = LNetStreamer_ConfigType_STANDARD;

                            eInnerStatus = LNetStreamer_GetConfig(hSession, &poSessionCfg->eConfigType, uiCfgLen);

                            if (LSTATUS_IS_SUCCESS(eInnerStatus))
                            {
                                poRtspSrvMod->szServerLocation[0] = '\0';
                                if (poSessionCfg->uiConfigFlags & LNetStreamer_ConfigFlags_LOCATION)
                                {
                                    strncpy_wz(poRtspSrvMod->szServerLocation,
                                               poSessionCfg->pacLocation,
                                               sizeof(poRtspSrvMod->szServerLocation));
                                }
                                poRtspSrvMod->bSessionStarted = MTRUE;
                            }

                            free(poSessionCfg);
                        }
                    }
                }

                if (bSessionInit)
                {
                    // Unstub the queue once the GetConfig is done since it can take several hundreds of milliseconds to execute.
                    poInLink->poModLnk->abStubOutput[poInLink->uiSubmitIdx] = MFALSE;
                }
            }

            if (poSrcBuffer != MNULL)
            {
                ModLnkIn_ReturnBuffer(poInLink, poSrcBuffer, MNULL, NO_TAG);
            }
        }

        if (LSTATUS_IS_FAIL(eStatus)
            && (eStatus != LStatus_TIMEOUT))
        {
            usleep(1000);
        }
    }

    if (oBufferInfo.uiDataMaxLength > 0)
    {
        oBufferInfo.uiDataLength = 0;

        LNetStreamer_ReleaseMedia(hSession, &oBufferInfo.eMediaType);
    }

    return eStatus;
}

/************************************************************************************************************\

Function:       RtspSrvMod_MainCpuThread

Description:    .

\************************************************************************************************************/
LStatus RtspSrvMod_MainCpuThread(void* pvData)
{    
    ModThread_SetName("RtspSrv");

    if (pvData == MNULL)
    {
        MsgLogErr("ERROR! NULL data.");
        return LStatus_INVALID_PARAM;
    }

    MsgLog(2, "Start thread %p.", pthread_self());

    LNetStreamer_Handle hSession        = MNULL;
    RtspServerModule*   poRtspSrvMod    = (RtspServerModule*)pvData;
    LNetStreamer_Config oSessionOpts;

    memset(&oSessionOpts, 0, sizeof(oSessionOpts));

    oSessionOpts.eConfigType            = LNetStreamer_ConfigType_STANDARD;
    oSessionOpts.eService               = LNetStreamer_Service_SERVER;
    oSessionOpts.eProtocol              = poRtspSrvMod->eProtocol;
    oSessionOpts.hDevice                = MNULL;    // Will not create LBuffers
    oSessionOpts.bEnableVideo           = poRtspSrvMod->bEnableVideo;
    oSessionOpts.uiVideoBufferCount     = LNETSTREAMER_VIDEO_BUFFER_COUNT_DEFAULT;
    oSessionOpts.uiVideoBufferLength    = poRtspSrvMod->uiMaxNaluSizeBytes;
    oSessionOpts.bEnableAudio           = poRtspSrvMod->bEnableAudio;
    oSessionOpts.uiAudioBufferCount     = LNETSTREAMER_AUDIO_BUFFER_COUNT_DEFAULT;
    oSessionOpts.uiAudioBufferLength    = poRtspSrvMod->uiMaxAudioBufSizeBytes;
    oSessionOpts.paucVideoHeaderSps     = MNULL;
    oSessionOpts.paucVideoHeaderPps     = MNULL;
    oSessionOpts.uiVideoHeaderSpsLength = 0;
    oSessionOpts.uiVideoHeaderPpsLength = 0;
    oSessionOpts.uiConfigFlags          = LNetStreamer_ConfigFlags_RTSP_FOLDER
                                          | LNetStreamer_ConfigFlags_RTSP_PORT;
    oSessionOpts.pacRtspFolder          = poRtspSrvMod->szFolderName;
    oSessionOpts.uiRtspFolderLength     = strlen(poRtspSrvMod->szFolderName);
    oSessionOpts.uiRtspPort             = 3049;

    if(poRtspSrvMod->bEnableAudio)
    {
        oSessionOpts.uiConfigFlags |= LNetStreamer_ConfigFlags_AUDIO_HEADER;
        oSessionOpts.paucAudioHeader = (MUINT8*) &(poRtspSrvMod->uiAudioSpecificConfig);
        oSessionOpts.uiAudioHeaderLength = sizeof(poRtspSrvMod->uiAudioSpecificConfig);
    }

    if (poRtspSrvMod->szAddress[0])
    {
        oSessionOpts.uiConfigFlags |= LNetStreamer_ConfigFlags_DESTINATION;
        oSessionOpts.pacDestination = poRtspSrvMod->szAddress;
        oSessionOpts.uiDestinationLength = strlen(poRtspSrvMod->szAddress);
    }

    if (poRtspSrvMod->uiVideoPort)
    {
        oSessionOpts.uiConfigFlags |= LNetStreamer_ConfigFlags_VIDEO_PORT;
        oSessionOpts.uiVideoPort = poRtspSrvMod->uiVideoPort;
    }

    if (poRtspSrvMod->uiAudioPort)
    {
        oSessionOpts.uiConfigFlags |= LNetStreamer_ConfigFlags_AUDIO_PORT;
        oSessionOpts.uiAudioPort = poRtspSrvMod->uiAudioPort;
    }

    if (poRtspSrvMod->szRtmpPushLocation[0])
    {
        oSessionOpts.uiConfigFlags |= LNetStreamer_ConfigFlags_LOCATION;
        oSessionOpts.pacLocation = poRtspSrvMod->szRtmpPushLocation;
        oSessionOpts.uiLocationLength = strlen(poRtspSrvMod->szRtmpPushLocation);
    }

    if (poRtspSrvMod->szUsername[0] || poRtspSrvMod->szPassword[0])
    {
        oSessionOpts.uiConfigFlags |= LNetStreamer_ConfigFlags_RTSP_AUTH;
        oSessionOpts.pacUsername = poRtspSrvMod->szUsername;
        oSessionOpts.uiUsernameLength = strlen(poRtspSrvMod->szUsername);
        oSessionOpts.pacPassword = poRtspSrvMod->szPassword;
        oSessionOpts.uiPasswordLength = strlen(poRtspSrvMod->szPassword);
    }

    if ((LNetStreamer_Protocol_RTSP == poRtspSrvMod->eProtocol) ||
        (LNetStreamer_Protocol_RTP == poRtspSrvMod->eProtocol))
    {
        oSessionOpts.uiConfigFlags |= LNetStreamer_ConfigFlags_RTCP_FREQUENCY;
        oSessionOpts.uiRtcpFrequency = 1000;
    }

    if (poRtspSrvMod->uiMtu)
    {
        oSessionOpts.uiConfigFlags |= LNetStreamer_ConfigFlags_MTU;
        oSessionOpts.uiMtu = poRtspSrvMod->uiMtu;
    }

    if (poRtspSrvMod->bEnableSrt)
    {
        oSessionOpts.uiConfigFlags |= LNetStreamer_ConfigFlags_SRT;

        if (poRtspSrvMod->eSrtMode)
        {
            oSessionOpts.uiConfigFlags |= LNetStreamer_ConfigFlags_EXTRA_FLAGS;
            oSessionOpts.uiConfigFlags2 |= LNetStreamer_ConfigFlags2_SRT_MODE;
            oSessionOpts.eSrtMode = poRtspSrvMod->eSrtMode;
        }
    }

    if (poRtspSrvMod->uiHlsDuration)
    {
        oSessionOpts.uiConfigFlags |= LNetStreamer_ConfigFlags_EXTRA_FLAGS;
        oSessionOpts.uiConfigFlags2 |= LNetStreamer_ConfigFlags2_HLS_DURATION;
        oSessionOpts.uiHlsDuration = (MUINT32)(poRtspSrvMod->uiHlsDuration);
    }

    if (poRtspSrvMod->uiHlsCount)
    {
        oSessionOpts.uiConfigFlags |= LNetStreamer_ConfigFlags_EXTRA_FLAGS;
        oSessionOpts.uiConfigFlags2 |= LNetStreamer_ConfigFlags2_HLS_COUNT;
        oSessionOpts.uiHlsCount = (MUINT32)(poRtspSrvMod->uiHlsCount);
    }

    if (poRtspSrvMod->szHlsFolder[0u])
    {
        oSessionOpts.uiConfigFlags |= LNetStreamer_ConfigFlags_EXTRA_FLAGS;
        oSessionOpts.uiConfigFlags2 |= LNetStreamer_ConfigFlags2_HLS_FOLDER;
        oSessionOpts.pacHlsFolder = poRtspSrvMod->szHlsFolder;
        oSessionOpts.uiHlsFolderLength = (MUINT32)(strlen(poRtspSrvMod->szHlsFolder));
    }

    if (poRtspSrvMod->szHlsMaster[0u])
    {
        oSessionOpts.uiConfigFlags |= LNetStreamer_ConfigFlags_EXTRA_FLAGS;
        oSessionOpts.uiConfigFlags2 |= LNetStreamer_ConfigFlags2_HLS_MASTER;
        oSessionOpts.pacHlsMaster = poRtspSrvMod->szHlsMaster;
        oSessionOpts.uiHlsMasterLength = (MUINT32)(strlen(poRtspSrvMod->szHlsMaster));
    }

    if (poRtspSrvMod->szNetInterface[0u])
    {
        oSessionOpts.uiConfigFlags |= LNetStreamer_ConfigFlags_NET_INTERFACE;
        oSessionOpts.pacNetInterface = poRtspSrvMod->szNetInterface;
        oSessionOpts.uiNetInterfaceLength = (MUINT32)(strlen(poRtspSrvMod->szNetInterface));
    }

    if (poRtspSrvMod->bEnableIpv6)
    {
        oSessionOpts.uiConfigFlags |= LNetStreamer_ConfigFlags_EXTRA_FLAGS;
        oSessionOpts.uiConfigFlags2 |= LNetStreamer_ConfigFlags2_IPV6;
    }

    /* Allow TCP interleaved connections */
    oSessionOpts.uiConfigFlags |= LNetStreamer_ConfigFlags_EXTRA_FLAGS;
    oSessionOpts.uiConfigFlags2 |= LNetStreamer_ConfigFlags2_TCP_INTERLEAVED;

    LStatus eStatus = LNetStreamer_Create(&oSessionOpts.eConfigType, &hSession);

    if (LSTATUS_IS_FAIL(eStatus))
    {
        return eStatus;
    }

    StrmThreadData oAudioData;
    oAudioData.hSession         = hSession;
    oAudioData.poRtspSrvMod     = poRtspSrvMod;
    oAudioData.bIsVideoThread   = MFALSE;
    oAudioData.bIsMainThread    = MTRUE;

    if(poRtspSrvMod->bEnableVideo)
    {
        // Start child audio thread if needed
        if(poRtspSrvMod->bEnableAudio)
        {
            oAudioData.bIsMainThread = MFALSE;
            eStatus = ModThread_Start(&(poRtspSrvMod->oChildCpuThread), &oAudioData, RtspSrvMod_StreamingThread);
        }

        if(LSTATUS_IS_SUCCESS(eStatus))
        {
            StrmThreadData oVideoData;
            oVideoData.bIsMainThread  = MTRUE;
            oVideoData.bIsVideoThread = MTRUE;
            oVideoData.hSession       = hSession;
            oVideoData.poRtspSrvMod   = poRtspSrvMod;
            eStatus = RtspSrvMod_StreamingThread(&oVideoData);
        }
    }
    else if(poRtspSrvMod->bEnableAudio)
    {
        eStatus = RtspSrvMod_StreamingThread(&oAudioData);
    }
    else
    {
        eStatus =  LStatus_FAIL;
    }

    if(!oAudioData.bIsMainThread)
    {
        ModThread_Stop(&(poRtspSrvMod->oChildCpuThread));
    }

    poRtspSrvMod->bSessionStarted = MFALSE;

    if (hSession != MNULL)
    {
        LNetStreamer_Destroy(hSession);
    }

    MsgLog(2, "Kill thread.");

    return LStatus_OK;
}

/************************************************************************************************************\

Function:       RtspSrvMod_Start

Description:    .

\************************************************************************************************************/
LStatus RtspSrvMod_Start(RtspServerModule* poRtspSrvMod)
{
    MsgLog(2, "{...");

    LStatus eStatus = LStatus_INVALID_PARAM;

    if ((poRtspSrvMod != MNULL)
        && poRtspSrvMod->bInit)
    {
        eStatus = LStatus_OK;

        MBOOL32 bGoodConnection = MTRUE;

        if((poRtspSrvMod->bEnableVideo && (poRtspSrvMod->oVideoInLink.poModLnk == MNULL))
            || (poRtspSrvMod->bEnableAudio && (poRtspSrvMod->oAudioInLink.poModLnk == MNULL)))
        {
            bGoodConnection = MFALSE;
        }

        if (bGoodConnection)
        {
            eStatus = ModThread_Start(&(poRtspSrvMod->oMainCpuThread), poRtspSrvMod, RtspSrvMod_MainCpuThread);
        }

        else
        {
            MsgLogErr("ERROR! Bad connection.");
            eStatus = LStatus_FAIL;
        }
    }

    MsgLog(2, "...} (status= %d - %s)", eStatus, GetStatusStr(eStatus));

    return eStatus;
}

/************************************************************************************************************\

Function:       RtspSrvMod_Stop

Description:    .

\************************************************************************************************************/
void RtspSrvMod_Stop(RtspServerModule* poRtspSrvMod)
{
    MsgLog(2, "{...");

    if (poRtspSrvMod != MNULL)
    {
        ModThread_Stop(&(poRtspSrvMod->oMainCpuThread));
        poRtspSrvMod->uiStartTimeUsec = 0;
        poRtspSrvMod->uiLastAudioTsUsec = 0;
    }

    MsgLog(2, "...}");
}

